/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef RCM_CLASS_H
#define RCM_CLASS_H

#include <QString>
#include <QSerialPort>
#include <QTcpServer>
#include <QTcpSocket>
#include "miscellaneous.h"
#include "config_handler.h"
#include "rf_channel.h"

#if defined (Q_OS_LINUX)
#include "gpioclassv2.h"
#endif


class RCM_Class : public QObject
{
	Q_OBJECT

public:
	RCM_Class(QList<RF_Channel*> rf_channels);
	~RCM_Class();

	void RCM_kill();

private:
	QList<RF_Channel*> RF_Channels;
	ConfigHandler *config;

	QSerialPort *RCM_port = nullptr;
	QTcpServer *tcpServer = nullptr;
	QTcpSocket *tcpSocket = nullptr;
	QString ipAddress = "";		//IP Address of this system

	RCM_MODE last_RCM_mode = RCM_MODE::RCM_MODE_OFF;

	#if defined(Q_OS_LINUX)
		GPIOClass *GPIO_HAT_serial_mode;
//		GPIOClass *GPIO_RS485_RW_flipper;   //0 = read, 1 = transmit
	#endif

	// Serial Functions
	void Setup(QString portname);
	void Configure_Remote_Command_Mode(RCM_MODE mode);
	void Handler_RCM_Error();
	bool RCM_Open();

	// Serial + TCP
	void RCM_Close();
	void Disconnect_Signals_Slots();

	// TCP Functions
	bool RCM_TCP_Start_Listening();
	void Handler_RCM_TCP_Server_Error();
	void Handler_RCM_TCP_Socket_Error();
	void RCM_TCP_newConnection_Handler();
	void RCM_TCP_Client_Disconnected();
	void RCM_TCP_Kick_Client();

	// Generic communication functions
	void RCM_Write_Live(QString tx);
	QString	RCM_Read_Live();

	void RCM_Write_Blind(QString tx);
	QString RCM_Read_Blind();

private slots:
	void RCM_Blind_ReadyRead_Handler();
	void RCM_Blind_Response_from_SGX_Handler(QString response);

	void RCM_Live_ReadyRead_Handler();
	void RCM_Live_Response_from_SGX_Handler(QString response);

public slots:
	void Handler_Remote_Command_Mode_Set(RCM_MODE mode);

signals:
	/* Messaging signals */
	void signal_RCM_response(QString response);
	void signal_RCM_message(QString message);
	void signal_RCM_sweep_message(QString message);

	/* RCM state communication signals */
	void signal_RCM_error(QSerialPort::SerialPortError error, QString errorString, QString portName);
	void signal_RCM_notification(QString string);
	void signal_RCM_mode(RCM_MODE mode);

	/* RF Channel management signals */
	void signal_execute_reset_SGx(int intrasys_num);
	void signal_timerStop(int intrasys_num);
	void signal_timerStart(int intrasys_num, int polling_rate_ms);
	void signal_set_error_clearing_enable(int intrasys_num, bool enable);
};

#endif // RCM_CLASS_H
